function [m_x, m_fx] = vpowellxy(strF, x0, etol, phi, varargin),

% VPOWELLXY Implement a vectorized Powell's type method, that minimizes 
% over the (x, y) directions only.  
%
% VPOWELL2d(fStr, x0, etol, phi, ...). The objective fn is specified with   
% fStr, x0 = (x, y) are initial guesses, etol is final tolerance.  Here phi is 
% a matrix of parameters that will be reduced as searches complete.  Any 
% additional parameters are passed through on ther command line. 
%
% 
% IMPORTANT : 

% ... modify and re-cycle if not 
% etol  = (1e-6)/sqrt(2);  
cnt = 0; n = 0;
nd = (1:size(phi, 1))';

% initialize 
fx  = feval(strF, x0(:,1), phi, 1, varargin{:});
fx0 = fx;

% Implement the method 
while(cnt < 500),

    fprintf('Pass ... %d\n', cnt);
    
    % invalid function calls   
    i = ~isfinite(fx);
    if (any(i)),
        fprintf('(VPOWELLXY) Fn. call out-of-bounds.');
        m_x(nd(i),:) = x0(i,:);
        m_fx(nd(i))  = fx(i);
        if (all(i)),
            % early return
            return
        end 
        
        % reducing 
        i   = ~i; nd  = nd(i);        
        x0  = x0(i,:);
        phi = phi(i,:);
    end 
   
%    [ fx0 fx ]

    fx  = feval(strF, x0(:,1), phi, 1, varargin{:});
    fx0 = feval(strF, x0(:,2), phi, 2, varargin{:});
    i = ((abs(fx) < etol)&(abs(fx0) < etol));
    if (any(i)),
        fprintf('(VPOWELLXY) Error tolerance reached. OK.\n');
        m_x(nd(i), :)  = x0(i, :);    % this should be the target vector
        m_fx(nd(i))    = fx(i);
        
        if (all(i)), 
            % early return
            return
        end
        
        % reducing
        i   = ~i; nd  = nd(i);
        x0  = x0(i, :); fx  = fx(i); 
        phi = phi(i, :);
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % update next step (use unbounded method).
    
    fprintf('(VPOWELLXY) Looking for 1-D min.\n');
    
    % Find finite interval (VFINBKET)
    [a, b, fa, fb] = vfinbket(strF, x0(:,n+1), x0(:,n+1)+0.1, phi, n+1, varargin{:});    
    i = isnan(fa)&isnan(fb);
    if (any(i)),
        fprintf('(VPOWELLXY) Couldnt get FINITE bracket.\n');
        m_x(nd(i), :)  = x0(i, :);    % this should be the target vector
        m_fx(nd(i))    = nan;
        if (all(i)),
            return
        end
        % reduce 
        i = ~i; nd = nd(i);
        a = a(i); b = b(i);
        x0  = x0(i, :); fx  = fx(i); 
        phi = phi(i, :);
    end
    
    % Unbounded (vectorized) SECANT method
    fprintf('(VPOWELLXY) Try unbounded SECANT method.\n');
    [x0(:,n+1), fx] = vsecant(strF, a, b, etol, phi, n+1, varargin{:});
    phi(:,n+1) = x0(:,n+1); 
    n = ~n; cnt = cnt + 1;
end 

% cleanup
error('(VPOWELLXY) Ran out of iterations.');
return